package gov.va.med.mhv.usermgmt.main.service.adapter.impl;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.Socket;
import java.net.UnknownHostException;
import java.util.Date;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ca.uhn.hl7v2.HL7Exception;
import ca.uhn.hl7v2.llp.LLPException;
import gov.va.med.mhv.common.data.model.Patient;
import gov.va.med.mhv.usermgmt.main.service.adapter.LookupPatientAdapterWithMiddleNameService;

/**
 * Responsible for accessing patients from legacy Master Patient Index system.
 * New class (LookupPatientAdapterWithMiddleName) has been created so that the
 * existing class (LookupPatientAdapter) is not approved to be touched
 * 
 * @author DNS
 */
@Component
public class LookupPatientAdapterWithMiddleName implements LookupPatientAdapterWithMiddleNameService {

	private static final Log LOG = LogFactory.getLog(LookupPatientAdapterWithMiddleName.class);

	private static final String CRLF = "\r\n";

	private static final String CR = "\r";

	private static final String DATA_TERMINATOR = "\u001B\u001B\u001B";

	private static final String DATA_BEGIN_MARKER = "220 OK\r\nDATA PARAM=MPI\r\n";

	private static final String HELLO = "HELO";

	private static final String READY = "Ready for";

	private static final String SUCCESS = "200 OK";

	private static final String DATA_PARAM = "DATA PARAM=MPI";

	private static final String QUIT = "QUIT";

	private static final String TURN = "TURN";

	private static final String ZERO = "0";

	private static final int PREFIX_LENGTH = 3;

	private static final char CR_CHAR = '\r';

	private String host;

	private int port;

	private int timeout;

	private String request;

	private String response;

	private Socket socket = null;

	private String firstName;

	private String lastName;

	private String middleName;

	private String gender;

	private Date birthDate;

	private String ssn;

	private Long userId;

	@Autowired
	private MpiProperties mpiProperties;

	public LookupPatientAdapterWithMiddleName() {

	}

	public LookupPatientAdapterWithMiddleName(String firstName, String lastName, String middleName, String gender, Date birthDate, String ssn, Long userId) {
		if (LOG.isDebugEnabled()) {
			LOG.debug("Constructor(firstName:" + firstName + "; lastName:" + lastName + " middle name: " + middleName + "; birthDate=" + birthDate +
				" gender: " + gender + ")");
		}
		this.firstName = firstName;
		this.lastName = lastName;
		this.middleName = middleName;
		this.gender = gender;
		this.birthDate = birthDate;
		this.ssn = ssn;
		this.userId = userId;
	}

	public Patient lookupPatient()
		throws AdapterException, DuplicatePatientException {

		Patient patient = null;

		try {
			connect();

			request = HELLO + CRLF;
			OutputStream requestStream = socket.getOutputStream();
			send(request, requestStream);

			InputStream responseStream = socket.getInputStream();
			response = receive(responseStream, CRLF);

			if (response.indexOf(READY) != -1) {
				request = DATA_PARAM + CRLF;
				send(request, requestStream);

				request = new LookupPatientEncoderWithMiddleName().encode(firstName, lastName, middleName, gender, birthDate, ssn, userId);

				request = formatOutboundMessage(request);
				send(request, requestStream);
				response = receive(responseStream, CRLF);

				// MPI starts running query right after receiving DATAPARAM
				// block.
				// It reads TURN command after performing query when all results
				// are gathered.
				request = TURN + CRLF;
				send(request, requestStream);
				response = receive(responseStream, DATA_TERMINATOR);
				response = formatInboundMessage(response);

				request = SUCCESS + CRLF;
				send(request, requestStream);

				request = QUIT + CRLF;
				send(request, requestStream);

				patient = new LookupPatientDecoderWithMiddleName(firstName, lastName, middleName, gender, birthDate, ssn).decode(response);
			} else {
				request = QUIT + CRLF;
				send(request, requestStream);
			}
		} catch (UnknownHostException uhe) {
			throw new AdapterException(host, port, timeout);
		} catch (IOException ioe) {
			throw new AdapterException(host, port, timeout, request, response);
		} catch (LLPException llpe) {
			throw new AdapterException(host, port, timeout, request, response);
		} catch (HL7Exception hl7e) {
			throw new AdapterException(host, port, timeout, request, response, hl7e.getMessage());
		} finally {
			try {
				disconnect();
			} catch (IOException e) {
			}
		}

		return patient;
	}

	private void send(String msg, OutputStream requestStream)
		throws IOException {
		if (LOG.isDebugEnabled()) {
			LOG.debug("Sending [" + msg + "]");
		}
		requestStream.write(msg.getBytes());
		requestStream.flush();
	}

	private String receive(InputStream responseStream, String terminator)
		throws IOException {
		StringBuffer buffer = new StringBuffer();
		InputStreamReader reader = new InputStreamReader(responseStream);
		int i = 0;
		char[] arr = new char[512];
		while (i != -1 && buffer.indexOf(terminator) < 0) {
			i = reader.read(arr, 0, arr.length);
			if (i > 0) {
				buffer.append(arr, 0, i);
			}
		}

		String response = buffer.toString();
		if (LOG.isDebugEnabled()) {
			LOG.debug("Received [" + response + "]");
		}
		return response;
	}

	private String formatOutboundMessage(String msg)
		throws LLPException {
		String[] results = new String[4];
		String[] parts = msg.split(CR);
		if (parts.length != 3) {
			throw new LLPException("Query has to have exactly 3 segments.");
		}
		for (int i = 0; i < parts.length; i++) {
			results[i] = parts[i];
		}
		results[3] = DATA_TERMINATOR;
		StringBuffer message = new StringBuffer();
		for (int i = 0; i < results.length; i++) {
			String prefix = String.valueOf(results[i].length());
			while (prefix.length() < PREFIX_LENGTH) {
				prefix = ZERO + prefix;
			}

			message.append(prefix).append(results[i]);
		}
		return message.toString();
	}

	private String formatInboundMessage(String msg)
		throws LLPException {
		int count = 0;
		int marker = 0;
		StringBuffer result = new StringBuffer();
		if (!msg.startsWith(DATA_BEGIN_MARKER)) {
			throw new LLPException("MPI Data Begin marker is not found");
		}
		msg = msg.substring(DATA_BEGIN_MARKER.length());
		while (marker < msg.length()) {
			count = Integer.parseInt(msg.substring(marker, marker + PREFIX_LENGTH));
			if (count > 0) {
				String subString = msg.substring(marker + PREFIX_LENGTH, marker + PREFIX_LENGTH + count);
				if (!subString.equals(DATA_TERMINATOR)) {
					result.append(subString);
				}
			} else {
				if (result.charAt(result.length() - 1) != CR_CHAR) {
					result.append(CR);
				}
			}

			marker = marker + PREFIX_LENGTH + count;
		}

		return result.toString();
	}

	private void connect()
		throws UnknownHostException, IOException {
		host = mpiProperties.getHost();
		port = mpiProperties.getPort();
		timeout = mpiProperties.getTimeout();

		socket = new Socket(host, port);
		socket.setSoTimeout(timeout);
	}

	private void disconnect()
		throws IOException {
		if (socket != null && socket.isConnected()) {
			socket.close();
		}
	}

	public String getFirstName() {
		return firstName;
	}

	public void setFirstName(String firstName) {
		this.firstName = firstName;
	}

	public String getLastName() {
		return lastName;
	}

	public void setLastName(String lastName) {
		this.lastName = lastName;
	}

	public String getMiddleName() {
		return middleName;
	}

	public void setMiddleName(String middleName) {
		this.middleName = middleName;
	}

	public String getGender() {
		return gender;
	}

	public void setGender(String gender) {
		this.gender = gender;
	}

	public Date getBirthDate() {
		return birthDate;
	}

	public void setBirthDate(Date birthDate) {
		this.birthDate = birthDate;
	}

	public String getSsn() {
		return ssn;
	}

	public void setSsn(String ssn) {
		this.ssn = ssn;
	}

	public Long getUserId() {
		return userId;
	}

	public void setUserId(Long userId) {
		this.userId = userId;
	}

}
